clear all
close all
clc

%% Read output file
outputs = loadoutput( 'TRID_Helical Antenna_Helical Antenna\Helical Antenna.out' );
out = outputs( 1 );

%% Retreive samples information
[ samples, samples_count ] = out.samples;
samples = samples * 1e-6;

%% Plot far field gain data
[ theta, phi, ref_point ] = out.far_field_info();

theta_count = length( theta );
phi_count   = length( phi );

data = zeros( samples_count, phi_count );

for i = 1 : phi_count
    d = out.far_field_gain_data( 2, i, :, 'E', 'Total', 'Horizontal' );
    data( :, i ) = d;
end

figure( 'polar' )
title( 'Total Gain, Horizontal' )
polar( phi, abs( data( 1, : ) ) )

%% Plot voltage source current
figure( 'linear' )
title( 'Current in Antenna Port' )
xlabel( 'Frequency [MHz]' )
ylabel( 'Current [A]' )

data = out.segment_voltage_source_data( 1, :, 'Current' );
plot( samples, real( data ), '-o', samples, imag( data ), '--x' )

l = legend( 'Current Real', 'Current Image' );
l.visible = true;
l.location = 'TopRight';
l.textFont = 'verdana,12,0,1,0';
l.borderColor = [ 200 200 200 ];

%% Plot voltage source S11
figure( 'linear' )
title( 'S11' )
xlabel( 'Frequency [MHz]' )
ylabel( 'Reflection Coefficient [dB]' )

data = out.segment_voltage_source_data( 1, :, 'S11' );
plot( samples, 20 * log10( abs( data ) ), '-o' )

pause

%% Open Word document
w = actxserver( 'Word.Application' );
actxquery( w, 'Visible', 1 );
documents = actxquery( w, 'Documents' );
document = actxquery( documents, 'Add' );

%% Create header
selection = actxquery( w, 'Selection' );
actxquery(selection, 'Style', 'Title' );
actxquery(selection, 'TypeText', 'Project Name: Helical Antenna' );

%% Add time stamp
date = datetime;

selection = actxquery( w, 'Selection' );
actxquery( selection, 'TypeParagraph' );

font = actxquery( selection, 'Font' );
actxquery( font, 'Name', 'Arial' );
actxquery( font, 'Size', 14 );
actxquery( font, 'Bold', false );
actxquery( font, 'Italic', true );

actxquery(font, 'Color', 'wdColorGray35' );

paragraph_format = actxquery( selection, 'ParagraphFormat' );
actxquery( paragraph_format, 'Alignment', 'wdAlignParagraphLeft' );
actxquery( selection, 'TypeText', ['Date of report generation: ' date] );

%% Add text
selection = actxquery( w, 'Selection' );
actxquery( selection, 'TypeParagraph' );

font = actxquery( selection, 'Font' );
actxquery( font, 'Name', 'Arial' );
actxquery( font, 'Size', 14 );
actxquery( font, 'Bold', false );
actxquery( font, 'Italic', false );

actxquery( font, 'Color', 'wdColorBlack' );

paragraph_format = actxquery( selection, 'ParagraphFormat' );
actxquery( paragraph_format, 'Alignment', 'wdAlignParagraphLeft' );
actxquery( selection, 'TypeText', 'This example demonstrates a helical antenna with reflector. Calculation is performed from 250 MHz to 600 MHz with 2 MHz step. Antenna is fed by voltage source with magnitude 1 V.' );

%% Add table of parameters
selection = actxquery( w, 'Selection' );
range = actxquery( selection, 'Range' );

data_cell_p = {
    'Lowest frequency, [MHz]', '250';
    'Highest frequency, [MHz]', '600' ;
    'Voltage source magnitude, [V]', '1';
    'Feeding line impedance, [Ohm]', '50';
    'Length of helix, [m]', '1.32';
    'Diameter of helix, [m]', '0.24';
    'Number of turns', '10' };

nr_rows_p = size( data_cell_p, 1 );
nr_cols_p = size( data_cell_p, 2 );

tables = actxquery( document, 'Tables' );
table = actxquery( tables, 'Add', range, nr_rows_p, nr_cols_p, 1, 1 );
actxquery( table, 'Select' );

for r = 1:nr_rows_p
    for c = 1:nr_cols_p
        if c == 1
            selection = actxquery( w, 'Selection' );
            font = actxquery( selection, 'Font' );
            actxquery( font, 'Name', 'Arial' );
            actxquery( font, 'SetSize', 12 );
            actxquery( font, 'Bold', true );
            actxquery( font, 'Italic', true );
            actxquery( font, 'Color', 'wdColorBlack' );
            actxquery( selection, 'TypeText', data_cell_p{r,c} );
        else
            selection = actxquery( w, 'Selection' );
            font = actxquery( selection, 'Font' );
            actxquery( font, 'Name', 'Arial' );
            actxquery( font, 'Size', 12 );
            actxquery( font, 'Bold', false );
            actxquery( font, 'Italic', false );
            actxquery( font, 'Color', 'wdColorBlack' );
            actxquery( selection, 'TypeText', data_cell_p{r,c} );
        end
        actxquery( selection, 'MoveRight' );
    end
    actxquery(selection, 'MoveRight' );
end

selection = actxquery( w, 'Selection' );
font = actxquery( selection, 'Font' );
actxquery( font, 'Name', 'Arial' );
actxquery( font, 'SetSize', 10 );
actxquery( selection, 'TypeParagraph' );

%% Add table
g = getall( 'Figure' );
nr_rows_p = length( g );
nr_cols_p = 1;
selection = actxquery( w, 'Selection' );
range = actxquery( selection, 'Range' );
tables = actxquery( document, 'Tables' );
table = actxquery( tables, 'Add', range, 2*nr_rows_p, nr_cols_p, 1, 1 );

%% Add first plot
f = g( 1 );
print( f, '-dmeta' );
selection = actxquery( w, 'Selection' );
actxquery( selection, 'Paste' );
paragraph_format = actxquery( selection, 'ParagraphFormat' );
actxquery( paragraph_format, 'Alignment', 'wdAlignParagraphCenter' );
actxquery( selection, 'MoveDown' );

selection = actxquery( w, 'Selection' );
font = actxquery( selection, 'Font' );
actxquery( font, 'Name', 'Arial' );
actxquery( font, 'Size', 12 );
actxquery( font, 'Bold', false );
actxquery( font, 'Italic', false );
actxquery( font, 'Color', 'wdColorBlack' );
actxquery( selection, 'TypeText', [ 'Total Gain' ] );
paragraph_format = actxquery( selection, 'ParagraphFormat' );
actxquery( paragraph_format, 'Alignment', 'wdAlignParagraphCenter' );
actxquery( selection, 'MoveDown' );

%% Add second plot
f = g( 2 );
print( f, '-dmeta' );
selection = actxquery( w, 'Selection' );
actxquery( selection, 'Paste' );
paragraph_format = actxquery( selection, 'ParagraphFormat' );
actxquery( paragraph_format, 'Alignment', 'wdAlignParagraphCenter' );
actxquery( selection, 'MoveDown' );

selection = actxquery( w, 'Selection' );
font = actxquery( selection, 'Font' );
actxquery( font, 'Name', 'Arial' );
actxquery( font, 'Size', 12 );
actxquery( font, 'Bold', false );
actxquery( font, 'Italic', false );
actxquery( font, 'Color', 'wdColorBlack' );
actxquery( selection, 'TypeText', [ 'Current in Antenna Port' ] );
paragraph_format = actxquery( selection, 'ParagraphFormat' );
actxquery( paragraph_format, 'Alignment', 'wdAlignParagraphCenter' );
actxquery( selection, 'MoveDown' );

%% Add third plot
f = g( 3 );
print( f, '-dmeta' );
selection = actxquery( w, 'Selection' );
actxquery( selection, 'Paste' );
paragraph_format = actxquery( selection, 'ParagraphFormat' );
actxquery( paragraph_format, 'Alignment', 'wdAlignParagraphCenter' );
actxquery( selection, 'MoveDown' );

selection = actxquery( w, 'Selection' );
font = actxquery( selection, 'Font' );
actxquery( font, 'Name', 'Arial' );
actxquery( font, 'Size', 12 );
actxquery( font, 'Bold', false );
actxquery( font, 'Italic', false );
actxquery( font, 'Color', 'wdColorBlack' );
actxquery( selection, 'TypeText', [ 'Reflection Coefficient (S11) [dB]' ] );
paragraph_format = actxquery( selection, 'ParagraphFormat' );
actxquery( paragraph_format, 'Alignment', 'wdAlignParagraphCenter' );
actxquery( selection, 'MoveDown' );

%% Save document
actxquery( document, 'SaveAs', [ 'Report.docx' ] );
